// ================================
// Famly Image Downloader - background.js (MV3, optimiert & robust)
// ================================

if (globalThis.__FAMLY_BG_INIT__) {
  console.warn("[Famly DL] background already initialized – skipping");
} else {
  globalThis.__FAMLY_BG_INIT__ = true;

  // --------- Konstanten ----------
  const MAX_STORE_ITEMS = 5000; // weiches Limit für Persistenz
  const STORAGE_KEY = "famlyImages";
  const PARALLEL = 10;

  // --------- State ----------
  /** @type {Array<{url:string,key?:string,imageId?:string,createdAt?:string,filename:string,_k:string}>} */
  let images = [];
  const seenKeys = new Set(); // "id:<imageId>" oder "url:<normUrl>"

  // --------- Badge ----------
  function updateBadge() {
    chrome.action.setBadgeBackgroundColor({ color: "#4688F1" });
    chrome.action.setBadgeText({ text: images.length ? String(images.length) : "" });
  }

  // --------- Helpers ----------
  function zero(n){ return String(n).padStart(2,"0"); }
  function formatCreatedAt(s){
    try{
      const d = new Date(s);
      return `${d.getFullYear()}-${zero(d.getMonth()+1)}-${zero(d.getDate())}_${zero(d.getHours())}-${zero(d.getMinutes())}-${zero(d.getSeconds())}`;
    }catch{ return "unknown_date"; }
  }
  function sanitizeSegment(s){ return String(s ?? "").replace(/[\\/:*?"<>|]+/g,"_").trim(); }
  function extFromKeyOrUrl(s){
    const m = /(\.[a-zA-Z0-9]{3,4})(?:\?|$)/.exec(s || "");
    return m ? m[1].toLowerCase() : ".jpg";
  }
  function normUrl(u) {
    try { const x = new URL(u); x.search = ""; return x.toString(); }
    catch { return u || ""; }
  }
  function recKey(entry){
    if (entry?.imageId) return `id:${entry.imageId}`;
    return `url:${normUrl(entry?.url || "")}`;
  }
  function makeFilename(entry){
    const stamp = formatCreatedAt(entry.createdAt);
    const id = entry.imageId || "unknown_id";
    const ext = extFromKeyOrUrl(entry.key || entry.url || "");
    const d = entry.createdAt ? new Date(entry.createdAt) : null;
    const year = d ? d.getFullYear() : "misc";
    const yearMonth = d ? `${d.getFullYear()}-${zero(d.getMonth()+1)}` : "misc";
    return `Famly/${year}/${yearMonth}/${sanitizeSegment(stamp)}_${sanitizeSegment(id)}${ext}`;
  }

  // --------- Persistenz (debounced) ----------
  let saveTimer = null;
  function scheduleSave(){
    if (saveTimer) clearTimeout(saveTimer);
    saveTimer = setTimeout(() => {
      chrome.storage.local.set({ [STORAGE_KEY]: images.slice(0, MAX_STORE_ITEMS) })
        .catch?.(() => {});
    }, 250);
  }

  // Dedup + Hinzufügen
  function addImage(entry){
    if (!entry || !entry.url) return;
    const key = recKey(entry);
    if (seenKeys.has(key)) return;

    const rec = {
      url: entry.url,
      key: entry.key,
      imageId: entry.imageId,
      createdAt: entry.createdAt,
      filename: makeFilename(entry),
      _k: key
    };
    seenKeys.add(key);
    images.push(rec);
    scheduleSave();
    updateBadge();
  }

  // Beim Start: gespeicherte Liste laden
  chrome.storage.local.get([STORAGE_KEY], (res) => {
    const arr = res?.[STORAGE_KEY];
    if (Array.isArray(arr)) {
      images = arr;
      seenKeys.clear();
      for (const r of images) seenKeys.add(recKey(r));
      updateBadge();
    }
  });

  // ================================
  // Früh registriertes Main-World-Content-Script (pageHook.js)
  // ================================
  async function registerMainHook() {
    try {
      const existing = await chrome.scripting.getRegisteredContentScripts().catch(() => []);
      const has = (existing || []).some(s => s.id === "famly-main-hook");
      if (has) await chrome.scripting.unregisterContentScripts({ ids: ["famly-main-hook"] });

      await chrome.scripting.registerContentScripts([{
        id: "famly-main-hook",
        js: ["pageHook.js"],
        matches: ["https://app.famly.de/*"],
        runAt: "document_start",
        world: "MAIN",
        allFrames: true
      }]);
    } catch (e) {
      console.error("registerMainHook failed:", e);
    }
  }
  chrome.runtime.onInstalled.addListener(registerMainHook);
  chrome.runtime.onStartup.addListener(registerMainHook);

  // ================================
  // Downloads (Queue mit max. 10 parallel)
  // ================================
  async function runDownloads(list, maxParallel = PARALLEL) {
    // Sicherheits-Dedupe nach internem Key
    const byKey = new Map();
    for (const it of list) byKey.set(it._k || recKey(it), it);
    const uniqueList = Array.from(byKey.values());

    let idx = 0;
    async function worker(){
      while (idx < uniqueList.length) {
        const item = uniqueList[idx++];
        try {
          await new Promise((resolve) => {
            chrome.downloads.download({
              url: item.url,
              filename: item.filename,
              saveAs: false
            }, () => resolve());
          });
        } catch (e) {
          console.error("Download fehlgeschlagen:", item.url, e);
        }
      }
    }
    const workers = [];
    for (let i=0;i<Math.min(maxParallel, uniqueList.length);i++) workers.push(worker());
    await Promise.all(workers);
  }

  // ================================
  // CSV-Export
  // ================================
  function toCSV(rows){
    const header = ["imageId","createdAt","url","filename","key"];
    const esc = (v) => {
      const s = v == null ? "" : String(v);
      return /[",\n]/.test(s) ? `"${s.replace(/"/g,'""')}"` : s;
    };
    const lines = [header.join(",")];
    for (const r of rows) {
      lines.push([r.imageId, r.createdAt, r.url, r.filename, r.key].map(esc).join(","));
    }
    return lines.join("\n");
  }
  function nowStamp(){
    const d = new Date();
    return `${d.getFullYear()}${zero(d.getMonth()+1)}${zero(d.getDate())}_${zero(d.getHours())}${zero(d.getMinutes())}${zero(d.getSeconds())}`;
  }
  async function exportCSV(){
    // Sicherheits-Dedupe vor Export
    const byKey = new Map();
    for (const it of images) byKey.set(it._k || recKey(it), it);
    const unique = Array.from(byKey.values());

    const csv = toCSV(unique);
    const dataUrl = "data:text/csv;charset=utf-8," + encodeURIComponent(csv);
    await new Promise((resolve) => {
      chrome.downloads.download({
        url: dataUrl,
        filename: `Famly/exports/famly_images_${nowStamp()}.csv`,
        saveAs: false
      }, () => resolve());
    });
  }

  // ================================
  // Messages (einziger Listener!)
  // ================================
  chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    (async () => {
      if (msg.action === "ingestImages" && Array.isArray(msg.images)) {
        for (const it of msg.images) addImage(it);
        sendResponse?.({ ok: true, added: msg.images.length });
        return;
      }

      if (msg.action === "debugEvent") {
        // console.debug("[Famly DL dbg]", msg.payload);
        sendResponse?.({ ok: true });
        return;
      }

      if (msg.action === "getCount") {
        sendResponse({ count: images.length });
        return;
      }

      if (msg.action === "downloadFirstN") {
        const n = Math.max(0, Math.min(msg.n || 3, images.length));
        await runDownloads(images.slice(0, n), PARALLEL);
        sendResponse({ ok: true, downloaded: n });
        return;
      }

      if (msg.action === "downloadAll") {
        await runDownloads(images, PARALLEL);
        images = [];
        seenKeys.clear();
        chrome.storage.local.remove(STORAGE_KEY);
        updateBadge();
        sendResponse({ ok: true });
        return;
      }

      if (msg.action === "exportCSV") {
        await exportCSV();
        sendResponse({ ok: true });
        return;
      }
    })();
    return true; // async sendResponse
  });

  // initiale Badge-Anzeige
  updateBadge();
}
